<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;

use Filament\Models\Contracts\FilamentUser;
use Filament\Models\Contracts\HasName;
use Filament\Panel;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

use App\Models\SponsorButton;
use App\Models\SponsorSocial;
use App\Models\TrustedNetwork;
use App\Models\Sale;
use App\Models\Button;
use App\Models\Social;
use App\Models\Product;
use App\Models\Service;
use App\Models\Event;
use App\Models\Video;
use App\Models\Sponsor;
use App\Models\Review;
use App\Models\UserSettings;
use App\Models\SubCard;
use App\Models\Referral;
use App\Models\Replay;
use App\Models\ButtonAnalytics;
use App\Models\CardViews;
use App\Models\PresentationPage;
use App\Models\Image;
use App\Models\Ad;
use App\Settings\CardDefaultsSetting;
use App\Models\Traits\HasProfilePhoto;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use App\Helpers\SavingLinkHelpers;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Carbon\Carbon;
use Laravel\Cashier\Billable;
use App\Notifications\CustomVerifyEmail;
use App\Observers\UserObserver;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use App\Models\LeadCaptureForm;

#[ObservedBy([UserObserver::class])]

class User extends Authenticatable implements FilamentUser, HasName, MustVerifyEmail
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;
    use HasProfilePhoto;
    use Billable;

    public function canAccessPanel(Panel $panel): bool
    {
        return $this->is_admin === 1;
    }

    public function getFilamentName(): string
    {
        return $this->username;
    }

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'username',
        'firstname',
        'lastname',
        'phone',
        'gender',
        'profile_pic',
        'thumbnail',
        'cover_pic',
        // 'company_pic',
        'job_title',
        'bio',
        'bio_long',
        'company',
        'company_website',
        // 'language',
        // 'address',
        'state',
        'country',
        'user_type',
        'email',
        'password',
        'is_active',
        'is_admin',
        'email_verified_at',
        'stripe_id',
        'access',
        'custom_trial',
        'trial_ends_at',
        'sponsor_id',
        'partnership_level',
        'fcm_token'
    ];

    protected $casts = [
        'trial_ends_at' => 'datetime',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
        ];
    }

    public function sendEmailVerificationNotification()
    {
        $this->notify(new CustomVerifyEmail);
    }

    public function lead_capture_form()
    {
        return $this->hasMany(LeadCaptureForm::class, 'user_id');
    }

    public function buttons()
    {
        return $this->hasMany(Button::class);
    }

    public function trusted_network()
    {
        return $this->hasMany(TrustedNetwork::class, 'user_id');
    }

    public function replays()
    {
        return $this->hasMany(Replay::class, 'user_id');
    }

    public function reviews()
    {
        return $this->hasMany(Review::class, 'user_id');
    }

    public function pages()
    {
        return $this->hasMany(PresentationPage::class, 'user_id');
    }

    public function product_category()
    {
        return $this->hasMany(ProductCategory::class, 'user_id');
    }

    public function product_collection()
    {
        return $this->hasMany(ProductCollection::class, 'user_id');
    }

    public function sales()
    {
        return $this->hasMany(Sale::class);
    }

    public function members()
    {
        return $this->hasMany(Member::class);
    }

    // public function referrals()
    // {
    //     return $this->hasMany(Referral::class);
    // }

    public function products()
    {
        return $this->hasMany(Product::class);
    }

    public function services()
    {
        return $this->hasMany(Service::class);
    }

    public function events()
    {
        return $this->hasMany(Event::class)
            ->whereNull('page_id'); // or your actual column that differentiates them
    }

    public function socials()
    {
        return $this->hasMany(Social::class);
    }

    public function videos()
    {
        return $this->hasMany(Video::class);
    }

    public function settings()
    {
        return $this->hasMany(UserSettings::class);
    }

    public function subcards()
    {
        return $this->hasMany(SubCard::class);
    }

    public function analytics()
    {
        return $this->hasMany(Analytics::class);
    }

    public function ads()
    {
        return $this->hasMany(Ad::class);
    }

    public function images()
    {
        return $this->hasMany(Image::class, 'user_id', 'id');
    }

    public function directions()
    {
        return $this->hasMany(Direction::class, 'user_id', 'id');
    }

    public function sponsor()
    {
        return $this->belongsTo(Sponsor::class, 'sponsor_id', 'id');
    }

    public function referralsAsMain()
    {
        return $this->hasMany(Referral::class, 'referrer_id_main');
    }

    public function referralsAsSub()
    {
        return $this->hasMany(Referral::class, 'referrer_id_sub');
    }

    public function referral()
    {
        return $this->hasOne(Referral::class, 'referred_id');
    }

    protected static function booted()
    {
        static::saving(function ($user) {
            if (!$user->is_admin) {
                if ($user->company_website) {
                    try {
                        $securedLink = SavingLinkHelpers::saveSecureLink($user['company_website']);
                        $user['company_website'] = $securedLink;
                    } catch (\InvalidArgumentException $e) {
                        throw new \InvalidArgumentException('Invalid input. Please enter a valid URL/link.');
                    }
                }

                if ($user->phone) {
                    try {
                        $contactNumber = SavingLinkHelpers::saveContactNumber($user['phone']);
                        $user['phone'] = $contactNumber;
                    } catch (\InvalidArgumentException $e) {
                        throw new \InvalidArgumentException('Invalid input. Please enter a valid number.');
                    }
                }
            }
        });

        static::creating(function ($user) {
            $default = app(CardDefaultsSetting::class);

            if (!$user->is_admin) {
                // Ensure the password is set and hashed if not already done
                if (empty($user->password)) {
                    $user->password = Hash::make($default->default_password);
                }

                //partnership level
                if (in_array($user->user_type, ['Gold', 'Platinum'])) {
                    if (empty($user->partnership_level)) {
                        $user->partnership_level = 'default';
                    }
                } else {
                    $user->partnership_level = null;
                }

                // if admin created this card, automatically verify the email
                if (!$user->is_duplicated) {
                    if (auth()->user() && auth()->user()->is_admin) {
                        $user->email_verified_at = Carbon::now();

                        if ($user->access === 'Trial') {
                            $user->trial_ends_at = now()->addDays(7);
                        }

                        if ($user->user_type === NULL) {
                            $user->access = NULL;
                        }
                    }
                }
            }
        });

        static::updated(function ($user) {
            if (!$user->is_admin) {
                // if ($user->cover_pic == NULL) {
                //     $user->settings()
                //         ->where('setting_key', 'profile_layout')
                //         ->update(['setting_value' => 'default']);
                // }

                $layout = $user->settings()
                    ->where('setting_key', 'profile_layout')
                    ->value('setting_value');

                // If the layout is cover-based and there’s no cover_pic, reset to default
                if (in_array($layout, ['cover_only', 'default_cover']) && $user->cover_pic === null) {
                    $user->settings()
                        ->where('setting_key', 'profile_layout')
                        ->update(['setting_value' => 'default']);
                }
            }
        });

        static::updating(function ($user) {
            // if admin is updating this user
            // if (auth()->user() && auth()->user()->is_admin) {

            //     if ($user->user_type === NULL) {
            //         $user->access = NULL;
            //     }

            //     if ($user->access === 'Lifetime') {
            //         $user->trial_ends_at = NULL;
            //     }

            //     if ($user->access === 'Trial') {
            //         $user->trial_ends_at = now()->addDays(7);
            //     }
            // }

            //partnership level
            if (in_array($user->user_type, ['Gold', 'Platinum'])) {
                if (empty($user->partnership_level)) {
                    $user->partnership_level = 'default';
                }
            } else {
                $user->partnership_level = null;
            }

            if ($user->phone) {
                try {
                    $contactNumber = SavingLinkHelpers::saveContactNumber($user['phone']);
                    $user['phone'] = $contactNumber;
                } catch (\InvalidArgumentException $e) {
                    throw new \InvalidArgumentException('Invalid input. Please enter a valid number.');
                }
            }

            // if there's new profile pic, unlink old
            if ($user->isDirty('profile_pic') && $user->getOriginal('profile_pic')) {
                Storage::disk('public')->delete($user->getOriginal('profile_pic'));
            }

            // if there's new cover pic, unlink old
            if ($user->isDirty('cover_pic') && $user->getOriginal('cover_pic')) {
                Storage::disk('public')->delete($user->getOriginal('cover_pic'));
            }

            // if there's new thumbnail, unlink old
            if ($user->isDirty('thumbnail') && $user->getOriginal('thumbnail')) {
                Storage::disk('public')->delete($user->getOriginal('thumbnail'));
            }
        });

        static::created(function ($user) {

            $default = app(CardDefaultsSetting::class);
            $is_test_mode = $default->test_mode; // local variable, not $this

            $basic_yearly     = $is_test_mode ? $default->test_basic_yearly     : $default->live_basic_yearly;

            $basic_monthly    = $is_test_mode ? $default->test_basic_monthly    : $default->live_basic_monthly;

            $gold_yearly      = $is_test_mode ? $default->test_gold_yearly      : $default->live_gold_yearly;

            $gold_monthly     = $is_test_mode ? $default->test_gold_monthly     : $default->live_gold_monthly;

            $platinum_yearly  = $is_test_mode ? $default->test_platinum_yearly  : $default->live_platinum_yearly;

            $platinum_monthly = $is_test_mode ? $default->test_platinum_monthly : $default->live_platinum_monthly;

            $free_trial_duration = $default->free_trial_duration;

            // 🌟 Only if created by an admin
            // if (auth()->check() && auth()->user()->is_admin && $user->access === 'Year') {
            //     // $user->access = 'Year';
            //     $plan = request()->input('plan') ?? $user->user_type ?? 'Basic';
            //     $plan = ucfirst(strtolower($plan));

            //     $price = match ($plan) {
            //         'Basic'    => $basic_yearly,
            //         'Gold'     => $gold_yearly,
            //         'Platinum' => $platinum_yearly,
            //         default    => $basic_yearly,
            //     };

            //     try {
            //         $user->newSubscription('default', $price)
            //             ->trialDays(365)
            //             ->create();
            //     } catch (\Exception $e) {
            //         \Log::error("Failed to create trial subscription for user {$user->id}: {$e->getMessage()}");
            //     }
            // }

            // if (auth()->check() && auth()->user()->is_admin && $user->access === 'Month') {
            //     // $user->access = 'Month';
            //     $plan = request()->input('plan') ?? $user->user_type ?? 'Basic';
            //     $plan = ucfirst(strtolower($plan));

            //     $price = match ($plan) {
            //         'Basic'    => $basic_monthly,
            //         'Gold'     => $gold_monthly,
            //         'Platinum' => $platinum_monthly,
            //         default    => $basic_monthly,
            //     };

            //     try {
            //         $user->newSubscription('default', $price)
            //             ->trialDays(30)
            //             ->create();
            //     } catch (\Exception $e) {
            //         \Log::error("Failed to create trial subscription for user {$user->id}: {$e->getMessage()}");
            //     }
            // }

            // if (auth()->check() && auth()->user()->is_admin && $user->access === 'Custom') {
            //     // \Log::error("Access: {$user->access} | Days: {$user->custom_trial}");
            //     // $user->access = 'Custom';
            //     $plan = request()->input('plan') ?? $user->user_type ?? 'Basic';
            //     $plan = ucfirst(strtolower($plan));

            //     $price = match ($plan) {
            //         'Basic'    => $basic_monthly,
            //         'Gold'     => $gold_monthly,
            //         'Platinum' => $platinum_monthly,
            //         default    => $basic_monthly,
            //     };

            //     try {
            //         $user->newSubscription('default', $price)
            //             // ->trialDays($user->custom_trial)
            //             ->trialDays((int) $user->custom_trial)
            //             ->create();
            //     } catch (\Exception $e) {
            //         \Log::error("Failed to create trial subscription for user {$user->id}: {$e->getMessage()}");
            //     }
            // }

            if (in_array($user->access, ['Year', 'Month', 'Trial', 'Custom'])) {
                $trialDays = match ($user->access) {
                    'Year'   => 365,
                    'Month'  => 30,
                    'Custom' => (int) $user->custom_trial,
                    'Trial'  => (int) $free_trial_duration,
                    default  => 0,
                };

                if ($trialDays > 0) {
                    $user->trial_ends_at = now()->addDays($trialDays);
                    $user->save();
                }
            }

            if (!$user->is_duplicated && !$user->is_admin) {

                // If no community links, skip
                // if (! isset($default->community_links) || empty($default->community_links)) {
                //     return;
                // }

                // Loop each default community link
                if ($default->community_links) {
                    foreach ($default->community_links as $linkData) {
                        Button::create([
                            'user_id' => $user->id,
                            'title'   => $linkData['title'] ?? null,
                            'link'    => $linkData['link'] ?? null,
                            'type'    => $linkData['type'] ?? null,
                            'view'    => 'communities',
                            'page_id' => null,
                        ]);
                    }
                }

                // Now $user->id is available
                UserSettings::create([
                    'user_id' => $user->id,
                    'setting_key' => 'footer_enabled',
                    'setting_value' => true,
                    'page_id' => NULL,
                ]);

                UserSettings::create([
                    'user_id' => $user->id,
                    'setting_key' => 'footer_text',
                    'setting_value' => $default->footer_text,
                    'page_id' => NULL,
                ]);

                if (in_array($user->user_type, ['Gold', 'Platinum'])) {
                    UserSettings::create([
                        'user_id' => $user->id,
                        'setting_key' => 'footer_link',
                        'setting_value' => $default->site_url . 'register?ref=' . $user->username,
                        'page_id' => NULL,
                    ]);
                }

                //searchable set true by default
                UserSettings::create([
                    'user_id' => $user->id,
                    'setting_key' => 'searchable',
                    'setting_value' => true,
                    'page_id' => NULL,
                ]);

                //display country true by default
                UserSettings::create([
                    'user_id' => $user->id,
                    'setting_key' => 'display_country',
                    'setting_value' => true,
                    'page_id' => NULL,
                ]);

                //add to contact
                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Add to Contact',
                    'link' => '',
                    'type' => 'vcf',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                //How may I help?
                Button::create([
                    'user_id' => $user->id,
                    'title' => 'How may I help?',
                    'link' => '',
                    'type' => 'help',
                    'view' => 'links',
                    'page_id' => NULL,
                    'is_active' => false,
                ]);

                //Exchange Contacts
                // Button::create([
                //     'user_id' => $user->id,
                //     'title' => 'Exchange Contacts',
                //     'link' => '',
                //     'type' => 'exchange',
                //     'view' => 'links',
                //     'page_id' => NULL,
                // ]);

                Button::create([
                    'user_id' => $user->id,
                    'title' => "Let's Connect",
                    'link' => '',
                    'type' => 'exchange',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                //Send your feedback
                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Send your feedback',
                    'link' => '',
                    'type' => 'feedback',
                    'view' => 'links',
                    'page_id' => NULL,
                    'is_active' => false,
                ]);

                //My Bio
                Button::create([
                    'user_id' => $user->id,
                    'title' => 'My Bio',
                    'link' => '',
                    'type' => 'bio',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                //buttons
                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Email',
                    'link' => $user->email,
                    'type' => 'email',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                Button::create([
                    'user_id' => $user->id,
                    'title' => 'SMS/Text',
                    'link' => $user->phone,
                    'type' => 'sms',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Direct Call',
                    'link' => $user->phone,
                    'type' => 'phone',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                //contact points
                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Email',
                    'link' => $user->email,
                    'type' => 'email',
                    'view' => 'contacts',
                    'page_id' => NULL,
                ]);

                Button::create([
                    'user_id' => $user->id,
                    'title' => 'SMS/Text',
                    'link' => $user->phone,
                    'type' => 'sms',
                    'view' => 'contacts',
                    'page_id' => NULL,
                ]);

                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Direct Call',
                    'link' => $user->phone,
                    'type' => 'phone',
                    'view' => 'contacts',
                    'page_id' => NULL,
                ]);

                if ($user->company_website) {
                    Button::create([
                        'user_id' => $user->id,
                        'title' => 'Company Website',
                        'link' => $user->company_website,
                        'type' => 'link',
                        'view' => 'links',
                        'page_id' => NULL,
                    ]);
                }

                //Events
                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Events & Promotions',
                    'link' => '',
                    'type' => 'events',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                //Shop
                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Shop',
                    'link' => '',
                    'type' => 'shop',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                //Services
                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Services',
                    'link' => '',
                    'type' => 'services',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                //Sales Page
                // Button::create([
                //     'user_id' => $user->id,
                //     'title' => 'Sales',
                //     'link' => '',
                //     'type' => 'sales',
                //     'view' => 'links',
                //     'page_id' => NULL,
                // ]);

                //Meet the team
                // Button::create([
                //     'user_id' => $user->id,
                //     'title' => 'Meet the Team',
                //     'link' => '',
                //     'type' => 'team',
                //     'view' => 'links',
                //     'page_id' => NULL,
                // ]);

                //Meet the team
                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Driving Directions',
                    'link' => '',
                    'type' => 'driving',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Zoom Replays',
                    'link' => '',
                    'type' => 'zoom_replay',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Testimonials',
                    'link' => '',
                    'type' => 'reviews',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Trusted Network',
                    'link' => '',
                    'type' => 'trusted_network',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                Button::create([
                    'user_id' => $user->id,
                    'title' => 'Payment Links',
                    'link' => '',
                    'type' => 'payments',
                    'view' => 'links',
                    'page_id' => NULL,
                ]);

                if ($user->user_type == 'Sponsored') {
                    $sponsor_id = $user->sponsor_id;

                    $buttons = SponsorButton::where('is_active', true)
                        ->where('sponsor_id', $sponsor_id)
                        ->where('view', 'links')
                        ->orderBy('order', 'asc')
                        ->get();

                    //if user exist, get contact points
                    $contacts = SponsorButton::where('is_active', true)
                        ->where('sponsor_id', $sponsor_id)
                        ->where('view', 'contacts')
                        ->orderBy('order', 'asc')
                        ->get();

                    //if user exist, get socials
                    $socials = SponsorSocial::where('is_active', true)
                        ->where('sponsor_id', $sponsor_id)
                        ->orderBy('order', 'asc')
                        ->get();

                    if ($buttons->isNotEmpty()) {
                        foreach ($buttons as $button) {
                            Button::create([
                                'user_id' => $user->id,
                                'title' => $button->title,
                                'link' => $button->link,
                                'type' => $button->type,
                                'view' => $button->view,
                                'page_id' => NULL,
                            ]);
                        }
                    }

                    if ($contacts->isNotEmpty()) {
                        foreach ($contacts as $contact) {
                            Button::create([
                                'user_id' => $user->id,
                                'title' => $contact->title,
                                'link' => $contact->link,
                                'type' => $contact->type,
                                'view' => $contact->view,
                                'page_id' => NULL,
                            ]);
                        }
                    }

                    if ($socials->isNotEmpty()) {
                        foreach ($socials as $social) {
                            Social::create([
                                'user_id' => $user->id,
                                'link' => $social->link,
                                'type' => $social->type,
                                'page_id' => NULL,
                            ]);
                        }
                    }
                }
            }
        });

        static::deleting(function ($user) {
            //delete cover pic
            if ($user->cover_pic) {
                $path = 'app/public/' . $user->cover_pic;
                unlink(storage_path($path));
            }

            //delete profile pic
            if ($user->profile_pic) {
                $path = 'app/public/' . $user->profile_pic;
                unlink(storage_path($path));
            }

            //delete thumbnail
            if ($user->thumbnail) {
                $path = 'app/public/' . $user->thumbnail;
                unlink(storage_path($path));
            }

            //delete all product images if user is being deleted. that includes the products created in subcards
            $products = Product::where('user_id', $user->id)->get();
            if ($products) {
                foreach ($products as $product) {
                    // Delete product media
                    foreach ($product->media as $media) {
                        if ($media->type == 'image') {
                            $path = 'app/public/' . $media->path;
                            if (file_exists(storage_path($path))) {
                                unlink(storage_path($path));
                            }
                        }
                        $media->delete();
                    }

                    // Delete featured image
                    if ($product->featured_image) {
                        $path = 'app/public/' . $product->featured_image;
                        if (file_exists(storage_path($path))) {
                            unlink(storage_path($path));
                        }
                    }
                }
            }

            $events = Event::where('user_id', $user->id)->get();
            if ($events) {
                foreach ($events as $event) {
                    if ($event->exists && $event->image) {
                        $path = 'app/public/' . $event->image;
                        unlink(storage_path($path));
                    }
                }
            }

            //delete all service images if user is being deleted. that includes the services created in subcards
            $services = Service::where('user_id', $user->id)->get();
            if ($services) {
                foreach ($services as $service) {
                    if ($service->exists && $service->service_image) {
                        $path = 'app/public/' . $service->service_image;
                        unlink(storage_path($path));
                    }
                }
            }

            $subcards = SubCard::where('user_id', $user->id)->get();
            if ($subcards) {
                foreach ($subcards as $subcard) {
                    //delete all subcard profile pics if user is being deleted.
                    if ($subcard->profile_pic) {
                        $path = 'app/public/' . $subcard->profile_pic;
                        unlink(storage_path($path));
                    }

                    //delete thumbnail
                    if ($subcard->thumbnail) {
                        $path = 'app/public/' . $subcard->thumbnail;
                        unlink(storage_path($path));
                    }

                    //delete all product images if user is being deleted. that includes the products created in subcards
                    $products = Product::where('page_id', $subcard->id)->get();
                    if ($products) {
                        foreach ($products as $product) {
                            // Delete product media
                            foreach ($product->media as $media) {
                                if ($media->type == 'image') {
                                    $path = 'app/public/' . $media->path;
                                    if (file_exists(storage_path($path))) {
                                        unlink(storage_path($path));
                                    }
                                }
                                $media->delete();
                            }

                            // Delete featured image
                            if ($product->featured_image) {
                                $path = 'app/public/' . $product->featured_image;
                                if (file_exists(storage_path($path))) {
                                    unlink(storage_path($path));
                                }
                            }
                        }
                    }

                    //delete all service images if user is being deleted. that includes the services created in subcards
                    $services = Service::where('page_id', $subcard->id)->get();
                    if ($services) {
                        foreach ($services as $service) {
                            if ($service->exists && $service->service_image) {
                                $path = 'app/public/' . $service->service_image;
                                unlink(storage_path($path));
                            }
                        }
                    }

                    //delete all events images if user is being deleted. that includes the events created in subcards
                    $events = Event::where('page_id', $subcard->id)->get();
                    if ($events) {
                        foreach ($events as $event) {
                            if ($event->exists && $event->image) {
                                $path = 'app/public/' . $event->image;
                                unlink(storage_path($path));
                            }
                        }
                    }

                    //delete all subcard cover pics if user is being deleted.
                    if ($subcard->cover_pic) {
                        $path = 'app/public/' . $subcard->cover_pic;
                        unlink(storage_path($path));
                    }
                }
            }

            foreach ($user->subscriptions as $subscription) {
                try {
                    // Cancel on Stripe
                    $subscription->cancelNow();

                    // Delete subscription + items from your DB
                    $subscription->delete();
                } catch (\Exception $e) {
                    \Log::error("Failed to delete subscription for user {$user->id}: {$e->getMessage()}");
                }
            }
        });
    }
}
