<?php

namespace App\Observers;

use App\Models\LetsConnect;
use App\Models\User;
use App\Models\SubCard;
use App\Notifications\UserNotification;
use Filament\Notifications\Notification;
use App\Mail\OrderMail;
use Illuminate\Support\Facades\Mail;
use App\Settings\CardDefaultsSetting;
use App\Services\TwilioService;

class LetsConnectObserver
{
    public function created(LetsConnect $form): void
    {
        $recipient = User::where('is_admin', true)->get();

        // User Notification
        if (!empty($form->page_id)) {
            // Subcard
            $owner = SubCard::find($form->page_id);
            if (! $owner) {
                \Log::warning("SubCard not found: {$form->page_id}");
                return;
            }

            $owner->notify(
                new UserNotification(
                    title: "Form Submission: Let's Connect!",
                    message: "<span class='font-semibold'>{$form->firstname} {$form->lastname}</span> wants to connect with you. Please check the form submission.",
                    url: route('contact.view', $form->id)
                )
            );
        } elseif (!empty($form->user_id)) {
            // Main card user
            $owner = User::find($form->user_id);
            if (! $owner) return;

            if ($owner->is_admin) {
                return;
            }

            $owner->notify(
                new UserNotification(
                    title: "Form Submission: Let's Connect!",
                    message: "<span class='font-semibold'>{$form->firstname} {$form->lastname}</span> wants to connect with you. Please check the form submission.",
                    url: route('contact.view', $form->id)
                )
            );
        }

        $recordUrl = route(
            'filament.admin.resources.lets-connects.view',
            ['record' => $form->id]
        );

        Notification::make()
            ->title("Form Submission: Let's Connect!")
            ->body("A new 'Lets Connect' form submission from {$form->firstname} {$form->lastname} has been submitted. Please review the information provided.")
            ->actions([
                \Filament\Notifications\Actions\Action::make('view')
                    ->url($recordUrl)

            ])
            ->sendToDatabase($recipient);

        $default = app(CardDefaultsSetting::class);
        $twilio = new TwilioService();

        $message = "Hey admin!\nA new 'Lets Connect' form submission from {$form->firstname} {$form->lastname} has been received. Please review the information provided.\n"
            . "\n{$recordUrl}";


        // if turned on, admin will receive sms notification
        if ($default->admin_sms_notification && $default->admin_sms_number) {

            $numbers = collect(explode(',', $default->admin_sms_number))
                ->map(fn($n) => trim($n))
                ->filter();

            foreach ($numbers as $number) {
                try {
                    $twilio->sendSms($number, $message);
                } catch (\Exception $e) {
                    \Log::error("SMS failed: " . $e->getMessage());
                }
            }
        }

        // if turned on, admin will receive whatsapp notification
        if ($default->admin_whatsapp_notification && $default->admin_whatsapp_number) {

            $numbers = collect(explode(',', $default->admin_whatsapp_number))
                ->map(fn($n) => trim($n))
                ->filter();

            foreach ($numbers as $number) {
                try {
                    $twilio->sendWhatsapp($number, $message);
                } catch (\Exception $e) {
                    \Log::error("WhatsApp failed: " . $e->getMessage());
                }
            }
        }

        // if turned on, admin email(s) will receive email notification
        if ($default->admin_email_notification) {

            $emails = collect(explode(',', $default->admin_email))
                ->map(fn($email) => trim($email))
                ->filter(fn($email) => filter_var($email, FILTER_VALIDATE_EMAIL));

            Mail::to($emails)
                ->queue(
                    new OrderMail(
                        "Hey admin!<br><br>
                A new <strong>Lets Connect</strong> form submission from <strong>{$form->firstname} {$form->lastname}</strong> has been received. Please review the information provided.<br><br>
                <a href='{$recordUrl}' target='_blank'>{$recordUrl}</a>",
                        "Form Submission: Let's Connect!",
                        $default->sender_email
                    )
                );
        }
    }
}
