<?php

namespace App\Observers;

use App\Models\SendCard;
use App\Models\User;
use Filament\Notifications\Notification;
use App\Mail\OrderMail;
use Illuminate\Support\Facades\Mail;
use App\Settings\CardDefaultsSetting;
use App\Services\TwilioService;

class SendCardObserver
{
    public function created(SendCard $sendCard): void
    {
        $recipient = User::where('is_admin', true)->get();

        $recordUrl = route(
            'filament.admin.resources.send-cards.view',
            ['record' => $sendCard->id]
        );

        Notification::make()
            ->title('New form submission: Send Business Card')
            ->body("{$sendCard->firstname} {$sendCard->lastname} just submitted his/her card. Click to view full details.")
            ->actions([
                \Filament\Notifications\Actions\Action::make('view')
                    // ->url(route('filament.admin.resources.send-cards.index', $sendCard->id))
                    ->url($recordUrl)

            ])
            ->sendToDatabase($recipient);

        $default = app(CardDefaultsSetting::class);
        $twilio = new TwilioService();

        $message = "Hey admin!\n{$sendCard->firstname} {$sendCard->lastname} just submitted his/her card.\n"
            . "View full details here: \n{$recordUrl}";

        // if turned on, admin will receive sms notification
        if ($default->admin_sms_notification && $default->admin_sms_number) {

            $numbers = collect(explode(',', $default->admin_sms_number))
                ->map(fn($n) => trim($n))
                ->filter();

            foreach ($numbers as $number) {
                try {
                    $twilio->sendSms($number, $message);
                } catch (\Exception $e) {
                    \Log::error("SMS failed: " . $e->getMessage());
                }
            }
        }

        // if turned on, admin will receive whatsapp notification
        if ($default->admin_whatsapp_notification && $default->admin_whatsapp_number) {

            $numbers = collect(explode(',', $default->admin_whatsapp_number))
                ->map(fn($n) => trim($n))
                ->filter();

            foreach ($numbers as $number) {
                try {
                    $twilio->sendWhatsapp($number, $message);
                } catch (\Exception $e) {
                    \Log::error("WhatsApp failed: " . $e->getMessage());
                }
            }
        }

        // if turned on, admin email(s) will receive email notification
        if ($default->admin_email_notification) {

            $emails = collect(explode(',', $default->admin_email))
                ->map(fn($email) => trim($email))
                ->filter(fn($email) => filter_var($email, FILTER_VALIDATE_EMAIL));

            Mail::to($emails)
                ->queue(
                    new OrderMail(
                        "Hey admin!<br><br>
                {$sendCard->firstname} {$sendCard->lastname} just submitted his/her card. Click to view full details. <br><br>
                <a href='{$recordUrl}' target='_blank'>View card</a>
                ",
                        'New form submission: Send Business Card',
                        $default->sender_email
                    )
                );
        }
    }

    /**
     * Handle the SendCard "updated" event.
     */
    public function updated(SendCard $sendCard): void
    {
        //
    }

    /**
     * Handle the SendCard "deleted" event.
     */
    public function deleted(SendCard $sendCard): void
    {
        //
    }

    /**
     * Handle the SendCard "restored" event.
     */
    public function restored(SendCard $sendCard): void
    {
        //
    }

    /**
     * Handle the SendCard "force deleted" event.
     */
    public function forceDeleted(SendCard $sendCard): void
    {
        //
    }
}
