<?php

namespace App\Observers;

use App\Models\UpgradeRequest;
use App\Models\User;
use Filament\Notifications\Notification;
use App\Mail\OrderMail;
use Illuminate\Support\Facades\Mail;
use App\Settings\CardDefaultsSetting;
use App\Services\TwilioService;

class UpgradeRequestObserver
{
    public function created(UpgradeRequest $request): void
    {
        $recipients = User::where('is_admin', true)->get();

        $user = User::find($request->user_id);

        if (! $user) {
            return;
        }

        $recordUrl = route(
            'filament.admin.resources.upgrade-requests.view',
            ['record' => $request->id],
            true
        );

        Notification::make()
            ->title('New card upgrade request')
            ->body("{$user->firstname} {$user->lastname} is requesting a card upgrade. Click to view full details.")
            ->actions([
                \Filament\Notifications\Actions\Action::make('view')
                    ->url($recordUrl)
                    ->openUrlInNewTab()
            ])
            ->sendToDatabase($recipients);

        $default = app(CardDefaultsSetting::class);
        $twilio = new TwilioService();

        $message = "Hey admin!\n{$user->firstname} {$user->lastname} is requesting a card upgrade.\n"
            . "View full details here: \n{$recordUrl}";

        // if turned on, admin will receive sms notification
        if ($default->admin_sms_notification && $default->admin_sms_number) {

            $numbers = collect(explode(',', $default->admin_sms_number))
                ->map(fn($n) => trim($n))
                ->filter();

            foreach ($numbers as $number) {
                try {
                    $twilio->sendSms($number, $message);
                } catch (\Exception $e) {
                    \Log::error("SMS failed: " . $e->getMessage());
                }
            }
        }

        // if turned on, admin will receive whatsapp notification
        if ($default->admin_whatsapp_notification && $default->admin_whatsapp_number) {

            $numbers = collect(explode(',', $default->admin_whatsapp_number))
                ->map(fn($n) => trim($n))
                ->filter();

            foreach ($numbers as $number) {
                try {
                    $twilio->sendWhatsapp($number, $message);
                } catch (\Exception $e) {
                    \Log::error("WhatsApp failed: " . $e->getMessage());
                }
            }
        }

        // if turned on, admin email(s) will receive email notification
        if ($default->admin_email_notification) {

            $emails = collect(explode(',', $default->admin_email))
                ->map(fn($email) => trim($email))
                ->filter(fn($email) => filter_var($email, FILTER_VALIDATE_EMAIL));

            Mail::to($emails)
                ->queue(
                    new OrderMail(
                        "Hey admin!<br><br>
                {$user->firstname} {$user->lastname} is requesting a card upgrade. Click the link below to view full details. <br><br>
                <a href='{$recordUrl}' target='_blank'>View request</a>
                ",
                        'New card upgrade request',
                        $default->sender_email
                    )
                );
        }
    }
}
