<?php

namespace App\Observers;

use App\Models\User;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Auth;
use App\Mail\OrderMail;
use Illuminate\Support\Facades\Mail;
use App\Settings\CardDefaultsSetting;
use Illuminate\Support\Facades\Validator;
use App\Services\TwilioService;

class UserObserver
{
    public function created(User $user): void
    {
        // Skip if someone is logged in (admin creating user via Filament, seeder, etc.)
        if (Auth::check()) {
            return;
        }

        // Only run if this was a self-registration (no logged-in user)
        $recipients = User::where('is_admin', true)->get();

        $recordUrl = route(
            'filament.admin.resources.clients.edit',
            ['record' => $user->id]
        );

        Notification::make()
            ->title('New user registration')
            ->body("{$user->firstname} {$user->lastname} ({$user->email}) just registered. Click to view full details.")
            ->actions([
                \Filament\Notifications\Actions\Action::make('view')
                    ->url($recordUrl)
            ])
            ->sendToDatabase($recipients);

        $default = app(CardDefaultsSetting::class);

        // if turned on, admin email(s) will receive email notification
        if ($default->admin_email_notification) {

            $emails = collect(explode(',', $default->admin_email))
                ->map(fn($email) => trim($email))
                ->filter(fn($email) => filter_var($email, FILTER_VALIDATE_EMAIL));

            Mail::to($emails)
                ->queue(
                    new OrderMail(
                        "Hey admin!<br><br>
                {$user->firstname} {$user->lastname} ({$user->email}) just registered. Click the link to view full details. <br><br>
                <a href='{$recordUrl}' target='_blank'>View on Dashboard</a>
                ",
                        "New user registration",
                        $default->sender_email
                    )
                );
        }

        $twilio = new TwilioService();
        $message = "Hey admin!\n{$user->firstname} {$user->lastname} ({$user->email}) just registered.\n"
            . "View full details here: \n{$recordUrl}";

        // if turned on, admin will receive sms notification
        if ($default->admin_sms_notification && $default->admin_sms_number) {

            $numbers = collect(explode(',', $default->admin_sms_number))
                ->map(fn($n) => trim($n))
                ->filter();

            foreach ($numbers as $number) {
                try {
                    $twilio->sendSms($number, $message);
                } catch (\Exception $e) {
                    \Log::error("SMS failed: " . $e->getMessage());
                }
            }
        }

        // if turned on, admin will receive whatsapp notification
        if ($default->admin_whatsapp_notification && $default->admin_whatsapp_number) {

            $numbers = collect(explode(',', $default->admin_whatsapp_number))
                ->map(fn($n) => trim($n))
                ->filter();

            foreach ($numbers as $number) {
                try {
                    $twilio->sendWhatsapp($number, $message);
                } catch (\Exception $e) {
                    \Log::error("WhatsApp failed: " . $e->getMessage());
                }
            }
        }
    }

    public function updated(User $user): void
    {
        //
    }

    /**
     * Handle the User "deleted" event.
     */
    public function deleted(User $user): void
    {
        //
    }

    /**
     * Handle the User "restored" event.
     */
    public function restored(User $user): void
    {
        //
    }

    /**
     * Handle the User "force deleted" event.
     */
    public function forceDeleted(User $user): void
    {
        //
    }
}
