<?php

namespace App\View\Components;

use Illuminate\View\Component;
use Symfony\Component\Intl\Countries;
use Symfony\Component\Intl\Exception\MissingResourceException;

class CountryFlag extends Component
{
    public $country;
    public $state;
    public $flag;

    public function __construct($country, $state = null)
    {
        $this->country = $country;
        $this->state = $state;

        $this->flag = '';

        if (!empty($country)) {
            $isoCode = $this->normalizeCountryCode($country);

            if ($isoCode) {
                $this->flag = $this->emojiFlag($isoCode);
            }
        }
    }

    /**
     * Normalize input into a valid 2-letter ISO code.
     */
    protected function normalizeCountryCode($country): ?string
    {
        $country = trim($country);

        // Already an ISO2 or ISO3 code
        if (strlen($country) === 2 && ctype_alpha($country)) {
            return strtoupper($country);
        }

        if (strlen($country) === 3 && ctype_alpha($country)) {
            try {
                return Countries::getAlpha2Code(strtoupper($country));
            } catch (MissingResourceException) {
                return null;
            }
        }

        // Try exact match
        try {
            return Countries::getAlpha2Code($country);
        } catch (MissingResourceException) {
            // Try case-insensitive / partial match
            $all = Countries::getNames();

            $countryLower = strtolower($country);
            foreach ($all as $code => $name) {
                if (strtolower($name) === $countryLower || str_contains(strtolower($name), $countryLower)) {
                    return $code;
                }
            }
        }

        return null;
    }

    /**
     * Convert ISO2 country code to flag emoji.
     */
    protected function emojiFlag(string $isoCode): string
    {
        $isoCode = strtoupper($isoCode);
        return mb_convert_encoding(
            '&#' . (127397 + ord($isoCode[0])) . ';&#' . (127397 + ord($isoCode[1])) . ';',
            'UTF-8',
            'HTML-ENTITIES'
        );
    }

    public function render()
    {
        return view('components.country-flag');
    }
}
